using Eurostep.D2M.Domain.Model;
using Eurostep.ModelFramework.Mapping;
using Eurostep.ModelFramework.Mapping.Testing;
using MappingFwkDemo.Mapper.FamilyToD2M;
using MappingFwkDemo.Mapper.FamilyToD2M.Mappings;
using MappingFwkDemo.Toolbox.Family;
using MappingFwkDemo.Toolbox.Family.Model;

namespace MappingFwkDemo.Tests.Mapping
{
    public class FamilyToD2MTestMapping<TMapping> : TestMapping<TMapping, string, IFamilyObject, FamilyRepository, string, SoftTypeBaseObject, SoftTypeRepository>
         where TMapping : IMapping<FamilyRepository, SoftTypeRepository>
    {
    }

    public class MapperTest
    {
        [Fact]
        public void TestMaleMapping()
        {
            var mapping = new FamilyToD2MTestMapping<MaleToPersonDefaultInMapping>();

            Male familyMale = new Male();
            familyMale.FirstName = "MaleFirstName";
            familyMale.LastName = "MaleLastName";
            mapping.Add(familyMale);

            SoftTypeRepository result = mapping.Map();

            Assert.Empty(mapping.Errors);

            var d2mPersons = result.FindByType<Eurostep.D2M.Domain.Model.Model.PersonNamespace.Input.defaultIn>();
            Eurostep.D2M.Domain.Model.Model.PersonNamespace.Input.defaultIn d2mPerson = d2mPersons.FirstOrDefault();

            Assert.NotNull(d2mPerson);
            Assert.StartsWith("Male", d2mPerson.id);
            Assert.Equal(familyMale.FirstName, d2mPerson.firstName);
            Assert.Equal(familyMale.LastName, d2mPerson.lastName);
        }

        [Fact]
        public async Task TestCsvMappingAsync()
        {
            // Family
            var familyDataFilePath = @"Data\demoData.csv";

            // D2M SoftType toolbox
            var d2mDataFilePath = @"output.json";

            // Run mapper
            using var inputFs = File.OpenRead(familyDataFilePath);
            using var outputFs = File.OpenWrite(d2mDataFilePath);
            await Map<FamilyToD2MMapper>(inputFs, "csv", outputFs);
        }

        [Fact]
        public async Task TestXlsxMappingAsync()
        {
            // Family
            var familyDataFilePath = @"Data\demo.xlsx";

            // D2M SoftType toolbox
            var d2mDataFilePath = @"output.json";

            // Run mapper
            using var inputFs = File.OpenRead(familyDataFilePath);
            using var outputFs = File.OpenWrite(d2mDataFilePath);
            await Map<FamilyToD2MMapper>(inputFs, "xlsx", outputFs);
        }

        static async Task Map<TMapper>(Stream source, string sourceType, FileStream target)
        where TMapper : IMapper
        {
            var mapper = Activator.CreateInstance<TMapper>();
            FamilyRepository sourceRepository = (FamilyRepository)mapper.CreateSourceRepository();
            SoftTypeRepository targetRepository = (SoftTypeRepository)mapper.CreateTargetRepository();

            source.Position = 0;

            sourceRepository.BeginCommit();
            sourceRepository.Import(source, sourceType);
            sourceRepository.Commit();

            sourceRepository.BeginCommit();
            sourceRepository.Commit();

            targetRepository.BeginCommit();
            await mapper.Mapping(sourceRepository, targetRepository);
            targetRepository.Commit();

            if (targetRepository is SoftTypeRepository) {
                ((SoftTypeRepository)targetRepository).WriteJson(target, false);
            }
        }
    }
}
