using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Eurostep.D2M.Domain.Model
{
    public static class SoftTypeValidator
    {
        public static void Validation(IEnumerable<SoftTypeBaseObject> instances)
        {
            if (instances == null)
                return;

            var resolvedSoftTypeInstances = new List<SoftTypeBaseObject>();
            foreach (var instance in instances)
            {
                if (resolvedSoftTypeInstances.Contains(instance))
                    continue;

                var visited = new Stack<object>();
                visited.Push(instance);
                Validate(instance, resolvedSoftTypeInstances, visited);
                visited.Pop();

                resolvedSoftTypeInstances.Add(instance);
            }
        }

        public static void Validate(object obj, List<SoftTypeBaseObject> resolvedInstances, Stack<object> visited)
        {
            Type objType = obj.GetType();
            PropertyInfo[] properties = objType.GetProperties();
            foreach (PropertyInfo property in properties)
            {
                var att = property.GetCustomAttributes(typeof(SoftTypePropertyAttribute), false);
                if (att == null || att.Length == 0)
                {
                    continue;
                }

                object propValue = property.GetValue(obj, null);
                SoftTypePropertyAttribute attribute = (SoftTypePropertyAttribute)att[0];

                if (attribute.Required)
                {
                    if (propValue == null)
                    {
                        throw new Exception("Error: " + attribute.ErrorMessage);
                    }
                }

                if (propValue == null)
                    continue;

                Type propertyType = Nullable.GetUnderlyingType(property.PropertyType) ?? property.PropertyType;
                if (propertyType.IsGenericType)
                {
                    if (typeof(IEnumerable).IsAssignableFrom(propertyType))
                    {
                        var count = ((ICollection)propValue).Count;

                        if (!attribute.CheckContainmentRule(count))
                        {
                            throw new Exception($"Validation Error: {property.Name} count: {count} violate the containment rule: [Lower: {attribute.LowerRule}, Upper: {attribute.UpperRule}]");
                        }

                        foreach (var item in (ICollection)propValue)
                        {
                            bool isSoftType = false;
                            if (propValue is SoftTypeBaseObject)
                            {
                                isSoftType = true;
                            }

                            if (isSoftType && resolvedInstances.Contains(item))
                                continue;

                            if (visited.Contains(item))
                                continue;

                            visited.Push(item);
                            Validate(item, resolvedInstances, visited);
                            visited.Pop();

                            if (isSoftType)
                                resolvedInstances.Add(item as SoftTypeBaseObject);
                        }
                    }
                }
                else if (propertyType.Assembly == objType.Assembly)
                {
                    // instance of the nested class
                    if (propertyType.IsNested)
                    {
                        if (!propertyType.IsEnum)
                        {
                            if (visited.Contains(propValue))
                                continue;

                            visited.Push(propValue);
                            Validate(propValue, resolvedInstances, visited);
                            visited.Pop();
                        }
                    }
                    else if (propValue is SoftTypeBaseObject)
                    {
                        if (resolvedInstances.Contains(propValue))
                            continue;
                        if (visited.Contains(propValue))
                            continue;

                        visited.Push(propValue);
                        Validate(propValue, resolvedInstances, visited);
                        visited.Pop();

                        resolvedInstances.Add(propValue as SoftTypeBaseObject);
                    }
                }
            }
        }
    }
}
