namespace Eurostep.D2M.Domain.Model
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using Newtonsoft.Json;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;

    class SoftTypeSerializer
    {
        public SoftTypeSerializer()
        {
        }

        public void Serialize(SoftTypeBaseObject[] dataSection, Stream stream)
        {
            var streamWriter = new StreamWriter(stream);
            var writer = new JsonTextWriter(streamWriter);
            writer.Formatting = Formatting.Indented;
            WriteDataSection(dataSection, writer);
            writer.Flush();
        }

        private class SoftTypeIntermediateInstance : IEquatable<SoftTypeIntermediateInstance>
        {
            private string _schameId;
            private string _schemaType;
            private SoftTypeBaseObject _baseInstance;

            public string SchemaId => _schameId;
            public string SchemaType => _schemaType;
            public SoftTypeBaseObject BaseInstance => _baseInstance;

            public SoftTypeIntermediateInstance(string schemaId, string schameType, SoftTypeBaseObject baseInstance)
            {
                _schameId = schemaId;
                _schemaType = schameType;
                _baseInstance = baseInstance;
            }

            public bool Equals(SoftTypeIntermediateInstance other)
            {
                return _baseInstance.Equals(other.BaseInstance);
            }
        }

        private void WriteDataSection(SoftTypeBaseObject[] dataSection, JsonTextWriter writer)
        {
            var instances = PreprocessComplexInstances(dataSection);
            writer.WriteStartObject();
            {
                writer.WritePropertyName("softTypes");
                writer.WriteStartArray();
                foreach (var instance in instances)
                {
                    WriteInstances(instance.Key, instance.Value, writer);
                }
                writer.WriteEndArray();
            }
            writer.WriteEndObject();
            writer.Flush();
        }

        private void WriteInstances(string softTypeId, List<SoftTypeIntermediateInstance> instances, JsonTextWriter writer)
        {
            writer.WriteStartObject();
            {
                writer.WritePropertyName("$id");
                writer.WriteValue(softTypeId);
                writer.WritePropertyName("instances");
                writer.WriteStartArray();
                foreach (var instan in instances)
                {
                    WriteInstance(instan.SchemaId, instan.SchemaType, instan.BaseInstance, writer);
                }
                writer.WriteEndArray();
            }
            writer.WriteEndObject();
            writer.Flush();
        }

        private void WriteInstance(string schemaId, string schemaType, SoftTypeBaseObject instance, JsonTextWriter writer)
        {
            writer.WriteStartObject();
            {
                writer.WritePropertyName("$id");
                writer.WriteValue(instance.Uid);
                writer.WritePropertyName((schemaType == "in") ? "$inputSchemaRef" : "$outputSchemaRef");
                writer.WriteValue(schemaId);
                writer.WritePropertyName("data");
                writer.WriteStartObject();
                WriteProperties(instance, writer);
                writer.WriteEndObject();
            }
            writer.WriteEndObject();
            writer.Flush();
        }

        public void WriteProperties(object obj, JsonTextWriter writer)
        {
            if (obj == null)
                return;

            Type objType = obj.GetType();
            PropertyInfo[] properties = objType.GetProperties();
            foreach (PropertyInfo property in properties)
            {
                object propValue = property.GetValue(obj, null);

                if (propValue == null)
                    continue;

                if (obj is SoftTypeBaseObject && (property.Name == "Uid" || property.Name == "SoftTypeName" || property.Name == "IsConnection"))
                    continue;

                Type propertyType = Nullable.GetUnderlyingType(property.PropertyType) ?? property.PropertyType;
                if (propertyType.IsGenericType)
                {
                    if (typeof(IEnumerable).IsAssignableFrom(propertyType))
                    {
                        if (((ICollection)propValue).Count == 0)
                            continue;

                        writer.WritePropertyName(property.Name);
                        writer.WriteStartArray();

                        foreach (var item in (ICollection)propValue)
                        {
                            writer.WriteStartObject(); // ToDo: Need to check the item type before WriteObject
                            WriteProperties(item, writer);
                            writer.WriteEndObject();
                        }
                        writer.WriteEndArray();
                    }
                    else
                    {
                        writer.WritePropertyName(StringUtil.CheckAndValidOidNameOfObjectId(objType, property.Name));
                        writer.WriteValue(propValue);
                    }
                }
                else if (propertyType == typeof(string))
                {
                    writer.WritePropertyName(StringUtil.CheckAndValidOidNameOfObjectId(objType, property.Name));
                    writer.WriteValue(propValue);
                }
                else if (propertyType == typeof(DateTime))
                {
                    DateTimeKind kind = ((DateTime)propValue).Kind;

                    if (kind != DateTimeKind.Utc)
                    {
                        throw new NotSupportedException(string.Format("Only UTC date format is supported, type for property {0} is {1}.", property.Name, kind.ToString()));
                    }

                    writer.WritePropertyName(property.Name);
                    writer.WriteValue(DateTime.SpecifyKind(DateTime.Parse(propValue.ToString()), DateTimeKind.Utc));
                }
                else if (propertyType.Assembly == objType.Assembly)
                {
                    if (propertyType.IsNested)
                    {
                        if (propertyType.IsEnum)
                        {
                            writer.WritePropertyName(property.Name);
                            if (NormalizedIdentifierLookupTable.TryGetOriginIdentifier(propValue.ToString(), out string origin))
                            {
                                writer.WriteValue(origin);
                            }
                            else
                            {
                                writer.WriteValue(propValue.ToString());
                            }
                        }
                        else
                        {
                            writer.WritePropertyName(property.Name);
                            writer.WriteStartObject();
                            WriteProperties(propValue, writer);
                            writer.WriteEndObject();
                        }
                    }
                    else
                    {
                        if (propValue is SoftTypeBaseObject)
                        {
                            // softType
                            writer.WritePropertyName(property.Name);
                            WriteSoftTypeProperty((SoftTypeBaseObject)propValue, writer);
                        }
                        else
                        {
                            // Predefined Types in Nova
                            if (propertyType.IsEnum)
                            {
                                writer.WritePropertyName(property.Name);
                                if (NormalizedIdentifierLookupTable.TryGetOriginIdentifier(propValue.ToString(), out string origin))
                                {
                                    writer.WriteValue(origin);
                                }
                                else
                                {
                                    writer.WriteValue(propValue.ToString());
                                }
                            }
                            else
                            {
                                writer.WritePropertyName(property.Name);
                                writer.WriteStartObject();
                                WriteProperties(propValue, writer);
                                writer.WriteEndObject();
                            }
                        }
                    }
                }
                else
                {
                    writer.WritePropertyName(property.Name);
                    writer.WriteValue(propValue);
                }
            }
        }

        private void WriteSoftTypeProperty(SoftTypeBaseObject instance, JsonTextWriter writer)
        {
            string softTypeId = null;
            string schemaId = null;
            string schemaType = null;
            if (!GetSoftTypeIdandSchemaInfo(instance, out softTypeId, out schemaId, out schemaType))
            {
                throw new Exception($"Error: Invalid SoftType Instance");
            }

            writer.WriteStartObject();
            if (instance.IsConnection)
            {
                WriteProperties(instance, writer);
            }
            else
            {
                writer.WritePropertyName("$softType");
                writer.WriteValue(softTypeId);
                writer.WritePropertyName("$instanceRef");
                writer.WriteValue(instance.Uid);
            }
            writer.WriteEndObject();
            writer.Flush();
        }

        private Dictionary<string, List<SoftTypeIntermediateInstance>> PreprocessComplexInstances(SoftTypeBaseObject[] dataSection)
        {
            var instanceCache = new HashSet<SoftTypeBaseObject>(dataSection.Where(d => !d.IsConnection && !d.GetType().IsNested).Distinct());
            foreach (var instance in dataSection)
            {
                CheckComplexInstance(instance, instanceCache, true);
            }

            var result = new Dictionary<string, List<SoftTypeIntermediateInstance>>();
            foreach (var instance in instanceCache)
            {
                string softtypeId = null;
                string schemaId = null;
                string schemaType = null;
                if (!GetSoftTypeIdandSchemaInfo(instance, out softtypeId, out schemaId, out schemaType))
                {
                    throw new Exception($"Error: Invalid SoftType Instance");
                }

                List<SoftTypeIntermediateInstance> valueList = null;
                if (!result.ContainsKey(softtypeId))
                {
                    valueList = new List<SoftTypeIntermediateInstance>();
                    result.Add(softtypeId, valueList);
                }

                if (valueList == null)
                {
                    if (!result.TryGetValue(softtypeId, out valueList))
                    {
                        throw new Exception($"Error: Can't find valid SoftType Instances");
                    }
                }

                var newValaue = new SoftTypeIntermediateInstance(schemaId, schemaType, instance);
                valueList.Add(newValaue);
            }

            return result;
        }

        private void CheckComplexInstance(object instance, HashSet<SoftTypeBaseObject> instancesCache, bool ignoreItself = false)
        {
            if (instance == null)
                return;

            Type objType = instance.GetType();
            if (!ignoreItself && instance is SoftTypeBaseObject && !objType.IsNested)
            {
                var bo = (SoftTypeBaseObject)instance;
                if (bo.IsConnection)
                {
                    // for a Connection type, it's only contains $oid property
                    // it will not added to the SoftTypeBaseObject list
                    // and attributes retrive can be skipped
                    return;
                }

                // this will handle the bidirectional reference case
                if (instancesCache.Contains(bo))
                {
                    return;
                }

                instancesCache.Add(bo);
            }

            PropertyInfo[] properties = objType.GetProperties();
            foreach (PropertyInfo property in properties)
            {
                object propValue = property.GetValue(instance, null);

                if (propValue == null)
                    continue;

                if (property.PropertyType.IsGenericType)
                {
                    if (typeof(IEnumerable).IsAssignableFrom(property.PropertyType))
                    {
                        foreach (var item in (ICollection)propValue)
                        {
                            CheckComplexInstance(item, instancesCache);
                        }
                    }
                }
                else if (property.PropertyType.Assembly == objType.Assembly)
                {
                    CheckComplexInstance(propValue, instancesCache);
                }
            }
        }

        private bool GetSoftTypeIdandSchemaInfo(SoftTypeBaseObject instance, out string softTypeId, out string schemaId, out string schemaType)
        {
            softTypeId = null;
            schemaId = null;
            schemaType = null;

            string typeName = instance.GetType().ToString();
            string[] parts = typeName.Split(new char[] { '.' });

            if (parts.Length < 3)
                return false;

            schemaId = parts[parts.Length - 1];
            switch (parts[parts.Length - 2])
            {
                case "Input": schemaType = "in"; break;
                case "Output": schemaType = "out"; break;
                default: return false;
            }
            softTypeId = StringUtil.GetSoftTypeIdFromSoftTypeNamespace(parts[parts.Length - 3]);

            return true;
        }
    }

    public static class StringUtil
    {
        public static string FirstLetterToUpper(string name)
        {
            return char.ToUpper(name[0]) + name.Substring(1);
        }

        public static string FirstLetterToLower(string name)
        {
            return char.ToLower(name[0]) + name.Substring(1);
        }

        public static string GetSoftTypeIdFromSoftTypeNamespace(string nameSpace)
        {
            return nameSpace.Replace("Namespace", "");
        }

        public static string CheckAndValidOidNameOfObjectId(Type type, string name)
        {
            if ((type.Name == "ObjectId" || type.GetInterface("IConnection") != null) && name == "oid")
            {
                return "$oid";
            }

            return name;
        }

        public static bool IsSameSchemaType(string left, string right)
        {
            var ls = left.Split('.');
            var rs = right.Split('.');
            return ls[ls.Length - 2] == rs[rs.Length - 2];
        }
    }
}
