using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Eurostep.D2M.Domain.Model
{
    class SoftTypeDeserializer
    {
        private readonly Dictionary<Tuple<string, string>, Tuple<JObject, string, string>> _instances = new Dictionary<Tuple<string, string>, Tuple<JObject, string, string>>();
        private readonly Dictionary<string, Dictionary<string, PrimaryInstance>> _primaryInstancesCache = new Dictionary<string, Dictionary<string, PrimaryInstance>>();
        public Dictionary<string, Dictionary<string, PrimaryInstance>> PrimaryInstancesCache => _primaryInstancesCache;

        public SoftTypeDeserializer()
        { }

        public IEnumerable<SoftTypeBaseObject> deserialize(Stream stream)
        {
            StreamReader reader = new StreamReader(stream);
            JObject data = JObject.Parse(reader.ReadToEnd());
            ParseAndExtractData(data);
            CreatePrimaryInstances();

            return BuildSoftTypeInstances();
        }

        private void CreatePrimaryInstances()
        {
            PrimaryInstance prInstance = null;
            foreach (var instance in _instances)
            {
                var key = instance.Key;
                var value = instance.Value;
                prInstance = new PrimaryInstance(key.Item1, key.Item2, value.Item1, value.Item2, value.Item3);

                if (!PrimaryInstancesCache.ContainsKey(prInstance.SoftTypeId))
                {
                    PrimaryInstancesCache.Add(prInstance.SoftTypeId, new Dictionary<string, PrimaryInstance>());
                }

                Dictionary<string, PrimaryInstance> primaInstanceLst = null;
                if (!PrimaryInstancesCache.TryGetValue(prInstance.SoftTypeId, out primaInstanceLst))
                {
                    throw new Exception($"Error: SoftTypeImport Instance Cache of Type {prInstance.SoftTypeId} not found");
                }

                if (primaInstanceLst.ContainsKey(prInstance.InstanceId))
                {
                    throw new Exception($"Error: SoftTypeImport Instance Uniqueness rule violation {prInstance.SoftTypeId}, {prInstance.InstanceId}");
                }

                primaInstanceLst.Add(prInstance.InstanceId, prInstance);
            }
        }

        public void ParseAndExtractData(JObject data)
        {
            JArray softTypes;
            if (JTokenImpl.TryGetValue(data, "softTypes", out softTypes) == false)
            {
                throw new Exception($"Error: SoftTypeImport MissingJsonProperty <softTypes> at {data.Path}");
            }

            foreach (JObject softType in softTypes)
            {
                string id;
                if (JTokenImpl.TryGetValue(softType, "$id", out id) == false)
                {
                    throw new Exception($"Error: SoftTypeImport MissingJsonProperty <$id> at {data.Path}");
                }

                JArray instances;
                if (JTokenImpl.TryGetValue(softType, "instances", out instances) == false)
                {
                    continue;
                }

                foreach (JObject instance in instances)
                {
                    string instanceId;
                    if (JTokenImpl.TryGetValue(instance, "$id", out instanceId) == false)
                    {
                        throw new Exception($"Error: SoftTypeImport MissingJsonProperty <$id> at {data.Path}");
                    }

                    string schemaId = null;
                    string schemaType = null;
                    if (JTokenImpl.TryGetValue(instance, "$inputSchemaRef", out schemaId))
                    {
                        schemaType = "Input";
                    }
                    else if (JTokenImpl.TryGetValue(instance, "$outputSchemaRef", out schemaId))
                    {
                        schemaType = "Output";
                    }
                    else
                    {
                        // No specified Schema ref is ok?
                        // throw new Exception($"Error: SoftTypeImport MissingJsonProperty <$inputSchemaRef> nor <$outputSchemaRef> at {data.Path}");
                        schemaType = "Input";
                        schemaId = "defaultIn";
                    }

                    JObject value;
                    if (JTokenImpl.TryGetValue(instance, "data", out value) == false)
                    {
                        throw new Exception($"Error: SoftTypeImport MissingJsonProperty <data> at {data.Path}");
                    }

                    AddInstance(id, instanceId, value, schemaId, schemaType);
                }
            }
        }

        public void AddInstance(string softType, string instanceId, JObject body, string schemaId, string schemaType)
        {
            Tuple<string, string> key = Tuple.Create(softType, instanceId);
            if (_instances.ContainsKey(key))
            {
                throw new Exception($"Error: SoftTypeImport NotFoundInstanceId {softType}, {instanceId}");
            }

            _instances.Add(key, Tuple.Create(body, schemaId, schemaType));
        }

        List<SoftTypeBaseObject> BuildSoftTypeInstances()
        {
            foreach (var instances in PrimaryInstancesCache)
            {
                foreach (var instance in instances.Value)
                {
                    if (instance.Value == null)
                    {
                        continue;
                    }

                    BuildProperties(instance.Value.Instance, instance.Value.Data);
                    instance.Value.IsResolved = true;
                }
            }

            return FindResolvedSoftTypeInstances();
        }

        public void BuildProperties(object obj, JObject data)
        {
            foreach (var JDataProperty in data.Properties())
            {
                var objType = obj.GetType();
                PropertyInfo info = Array.Find(objType.GetProperties(), p => StringUtil.CheckAndValidOidNameOfObjectId(objType, p.Name) == JDataProperty.Name);

                if (info == null)
                {
                    throw new Exception($"Error: SoftTypeImport Property {JDataProperty.Name} of Type {obj.GetType()} is not found");
                }

                switch (JDataProperty.Value.Type)
                {
                    case JTokenType.Array:
                        Type elementType = null;
                        if (info.PropertyType.IsGenericType && info.PropertyType.GetGenericTypeDefinition() == typeof(ICollection<>))
                        {
                            elementType = info.PropertyType.GetGenericArguments()[0]; // use this...
                        }
                        var list = (ICollection)typeof(List<>)
                                .MakeGenericType(elementType)
                                .GetConstructor(Type.EmptyTypes)
                                .Invoke(null);
                        info.SetValue(obj, list);
                        JArray elements = (JArray)JDataProperty.Value;
                        object elementObj = null;
                        for (int i = 0; i < elements.Count; i++)
                        {
                            try
                            {
                                elementObj = Activator.CreateInstance(elementType);
                            }
                            catch (Exception e)
                            {
                                throw new Exception($" Activator Error: {e.Message} [data]: {elements} of [Type] {elementType}");
                            }
                            ((IList)list).Add(elementObj);
                            BuildProperties(elementObj, (JObject)elements[i]); // ToDo: Check it element is JObject
                        }
                        break;
                    case JTokenType.Object:
                        string softTypeId = null;
                        string softTypeInstanceRef = null;
                        object propObj = null;
                        if (CheckSoftTypeProperty(JDataProperty.Value, out softTypeId, out softTypeInstanceRef))
                        {
                            if (!GetInstanceRefFromCache(softTypeId, softTypeInstanceRef, out propObj))
                            {
                                throw new Exception($"Error: SoftTypeImport Instance reference uid:{softTypeInstanceRef} of Type {softTypeId} is not found");
                            }
                        }
                        else
                        {
                            JObject propertyValue = (JObject)JDataProperty.Value;
                            if (propertyValue != null && propertyValue.Properties().Any())
                            {
                                try
                                {
                                    var typeToCreate = info.PropertyType;
                                    if (typeToCreate.IsAbstract)
                                    {
                                        // When a property is based on connection port didn't specify schema
                                        var assembly = typeof(SoftTypeBaseObject).Assembly;
                                        var schemaTypes = assembly.GetTypes().Where(t => !t.IsAbstract && typeToCreate.IsAssignableFrom(t));
                                        typeToCreate = schemaTypes.Where(t => StringUtil.IsSameSchemaType(t.FullName, obj.GetType().FullName)).FirstOrDefault();
                                    }
                                    propObj = Activator.CreateInstance(typeToCreate);
                                    if (propObj is SoftTypeBaseObject)
                                    {
                                        if (JDataProperty.Value is JObject)
                                        {
                                            var childen = ((JObject)JDataProperty.Value).Children();
                                            if (childen.Count() == 1 && childen.FirstOrDefault()?.ToObject<JProperty>()?.Name == "$oid")
                                                ((SoftTypeBaseObject)propObj).IsConnection = true;
                                        }
                                    }
                                }
                                catch (Exception e)
                                {
                                    throw new Exception($" Activator Error: {e.Message} [data]: {propertyValue} of [Type] {info.PropertyType}");
                                }
                                BuildProperties(propObj, (JObject)JDataProperty.Value);
                            }
                        }
                        info.SetValue(obj, propObj);
                        break;
                    case JTokenType.String:
                        Type propertyType = Nullable.GetUnderlyingType(info.PropertyType) ?? info.PropertyType;
                        if (propertyType.IsEnum)
                        {
                            string enumIdentifierName = JDataProperty.Value.ToString();
                            if (NormalizedIdentifierLookupTable.TryGetNormalizedIdentifier(enumIdentifierName, out var normalized))
                            {
                                enumIdentifierName = normalized;
                            }

                            try
                            {
                                var enumValue = Enum.Parse(propertyType, enumIdentifierName);
                                info.SetValue(obj, enumValue);
                            }
                            catch (Exception e)
                            {
                                throw new Exception($" Enum.Parse Error: {e.Message} [data]: {enumIdentifierName} of [Type] {propertyType}");
                            }
                        }
                        else
                        {
                            try
                            {
                                info.SetValue(obj, JDataProperty.Value.ToString());
                            }
                            catch (Exception e)
                            {
                                throw new Exception($" PropertyInfo.SetValue Error: {e.Message} [data]: {JDataProperty.Value.ToString()} of [Type] {propertyType}");
                            }
                        }
                        break;
                    case JTokenType.Boolean:
                        try
                        {
                            info.SetValue(obj, Boolean.Parse(JDataProperty.Value.ToString()));
                        }
                        catch (Exception e)
                        {
                            throw new Exception($" Boolean.Parse Error: {e.Message} [data]: {JDataProperty.Value.ToString()} of [Type] Boolean");
                        }
                        break;
                    case JTokenType.Date:
                        try
                        {
                            object propertyVal = DateTime.SpecifyKind(DateTime.Parse(JDataProperty.Value.ToString()), DateTimeKind.Utc);
                            info.SetValue(obj, propertyVal);
                        }
                        catch (Exception e)
                        {
                            throw new Exception($" Convert.ChangeType Error: {e.Message} [data]: {JDataProperty.Value.ToString()} of Date");
                        }
                        break;
                    case JTokenType.Integer:
                    case JTokenType.Float:
                        var targetType = IsNullableType(info.PropertyType) ? Nullable.GetUnderlyingType(info.PropertyType) : info.PropertyType;
                        try
                        {
                            object propertyVal = Convert.ChangeType(JDataProperty.Value.ToString(), targetType);
                            info.SetValue(obj, propertyVal);
                        }
                        catch (Exception e)
                        {
                            throw new Exception($" Convert.ChangeType Error: {e.Message} [data]: {JDataProperty.Value.ToString()} of [Type] {targetType}");
                        }
                        break;
                    default:
                        break;
                }
            }
        }

        public List<SoftTypeBaseObject> FindResolvedSoftTypeInstances()
        {
            List<SoftTypeBaseObject> softTypeInstances = new List<SoftTypeBaseObject>();

            foreach (var instances in PrimaryInstancesCache)
            {
                foreach (var instance in instances.Value)
                {
                    if (instance.Value == null)
                    {
                        continue;
                    }

                    if (instance.Value.IsResolved)
                    {
                        softTypeInstances.Add(instance.Value.Instance);
                    }
                }
            }

            return softTypeInstances;
        }

        private bool CheckSoftTypeProperty(JToken data, out string softTypeId, out string softTypeInstanceRef)
        {
            softTypeId = null;
            softTypeInstanceRef = null;

            if (!JTokenImpl.TryGetValue(data, "$softType", out softTypeId))
            {
                return false;
            }

            if (!JTokenImpl.TryGetValue(data, "$instanceRef", out softTypeInstanceRef))
            {
                return false;
            }

            return true;
        }

        private bool GetInstanceRefFromCache(string softTypeId, string softTypeInstanceRef, out object propObj)
        {
            propObj = null;

            if (softTypeId == null)
            {
                return false;
            }

            if (softTypeInstanceRef == null)
            {
                return false;
            }

            Dictionary<string, PrimaryInstance> softTypeInstances = null;
            if (!PrimaryInstancesCache.TryGetValue(softTypeId, out softTypeInstances))
            {
                return false;
            }

            PrimaryInstance instance = null;
            if (!softTypeInstances.TryGetValue(softTypeInstanceRef, out instance))
            {
                return false;
            }

            instance.IsReferred = true;
            propObj = instance.Instance;

            return true;
        }

        private static bool IsNullableType(Type type)
        {
            return type.IsGenericType && type.GetGenericTypeDefinition().Equals(typeof(Nullable<>));
        }
    }

    public static class JTokenImpl
    {
        public static bool Has(JToken token, string propertyName)
        {
            if (token.Type != JTokenType.Object)
            {
                return false;
            }
            JToken _;
            if (((JObject)token).TryGetValue(propertyName, out _) == false)
            {
                return false;
            }
            return true;
        }

        public static bool TryGetValue(JToken token, string propertyName, out JToken value)
        {
            if (token == null)
            {
                value = null;
                return false;
            }

            if (token.Type != JTokenType.Object)
            {
                value = null;
                return false;
            }

            return ((JObject)token).TryGetValue(propertyName, out value);
        }

        public static bool TryGetValue(JToken token, string propertyName, out string value)
        {
            value = null;
            if (token.Type != JTokenType.Object)
            {
                return false;
            }

            JToken t;
            if (((JObject)token).TryGetValue(propertyName, out t) == false)
            {
                return false;
            }

            if (t.Type != JTokenType.String)
            {
                return false;
            }
            value = (string)t;
            return true;
        }

        public static bool TryGetValue(JToken token, string propertyName, out bool value)
        {
            value = default(bool);
            if (token.Type != JTokenType.Object)
            {
                return false;
            }

            JToken t;
            if (((JObject)token).TryGetValue(propertyName, out t) == false)
            {
                return false;
            }

            if (t.Type != JTokenType.Boolean)
            {
                return false;
            }
            value = (bool)t;
            return true;
        }

        public static bool TryGetValue(JToken token, string propertyName, out JArray value)
        {
            value = null;
            if (token.Type != JTokenType.Object)
            {
                return false;
            }

            JToken t;
            if (((JObject)token).TryGetValue(propertyName, out t) == false)
            {
                return false;
            }

            if (t.Type != JTokenType.Array)
            {
                return false;
            }
            value = (JArray)t;
            return true;
        }

        public static bool TryGetValue(JToken token, string propertyName, out JObject value)
        {
            value = null;
            if (token.Type != JTokenType.Object)
            {
                return false;
            }

            JToken t;
            if (((JObject)token).TryGetValue(propertyName, out t) == false)
            {
                return false;
            }

            if (t.Type != JTokenType.Object)
            {
                return false;
            }
            value = (JObject)t;
            return true;
        }
    }

    internal class PrimaryInstance
    {
        private readonly string _softTypeId;
        private readonly string _instanceId;
        private readonly string _schemaId;
        private readonly string _schemaType;
        private readonly string _instanceTypeId;
        private readonly JObject _data;
        private readonly SoftTypeBaseObject _instance;
        private readonly PropertyInfo[] _propertyInfos;
        private bool _isResolved;
        private bool _isReferred;

        public SoftTypeBaseObject Instance => _instance;
        public bool IsResolved
        {
            get
            {
                return _isResolved;
            }
            set
            {
                _isResolved = value;
            }
        }

        public bool IsReferred
        {
            get
            {
                return _isReferred;
            }
            set
            {
                _isReferred = value;
            }
        }
        public string SoftTypeId => _softTypeId;
        public string InstanceId => _instanceId;
        public string SchemaId => _schemaId;
        public string SchemaType => _schemaType;
        public string InstanceTypeId => _instanceTypeId;
        public JObject Data => _data;

        private PrimaryInstance() { }

        public PrimaryInstance(string softTypeId, string instanceId, JObject data, string schemaId, string schemaType)
        {
            _isResolved = false;
            _isReferred = false;
            _softTypeId = softTypeId;
            _instanceId = instanceId;
            _schemaId = schemaId;
            _schemaType = schemaType;
            _data = data;
            _instanceTypeId = MakeInstanceTypeId();

            // create the primary softtype instance
            Type baseType = typeof(SoftTypeBaseObject);
            var type = baseType.Assembly.GetType($"{baseType.Namespace}.Model.{InstanceTypeId}");
            if (type == null)
            {
                throw new Exception($"Can't find {_instanceTypeId} in {baseType.Assembly.FullName}");
            }
            _instance = (SoftTypeBaseObject)Activator.CreateInstance(type);
            _instance.Uid = _instanceId;
            // and get the propertyInfo of the type
            _propertyInfos = type.GetProperties();
        }

        private string MakeInstanceTypeId()
        {
            return $"{_softTypeId}Namespace.{_schemaType}.{_schemaId}";
        }
    }
}
